/********************************************************************
 * (C) Copyright 1998 by Hewlett-Packard GmbH. All rights reserved. *
 ********************************************************************/

#include <typedefs.h>

#include <termios.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <fcntl.h>

#include "b_io.h"
#include "b_serial.h"
#include "lasterr.h"

/* --------------------------------------------------------------------------
 * STATICS
 * -------------------------------------------------------------------------- */

/******************************************************************************/
static b_errtype BestClearCommError(b_portnumtype OsHandle)
{
  printf("Serial: ClearCommError\n");
  return B_E_ERROR;
}


/* --------------------------------------------------------------------------
 * Should only be called from BestDeviceConnect()
 * We set our DTR (card's DSR) to signal connect.
 * -------------------------------------------------------------------------- */

b_errtype BestSerDeviceConnect(b_portnumtype OsHandle)
{
  int minfo;

  printf("Serial: SerDeviceConnect\n");

  if (OsHandle<0)
    return B_E_BAD_HANDLE;

  /* start with a clean slate */
  (void) BestClearCommError(OsHandle);

  ioctl(OsHandle,TIOCMGET,&minfo);
  minfo |= TIOCM_DTR;
  ioctl(OsHandle,TIOCMSET, &minfo );
  
  DBG_PRINT("\nSerial Port Connected");

  return B_E_OK;
}

/* --------------------------------------------------------------------------
 * Should only be called from BestCheckConnection()
 * Card sets its DTR (our DSR) to signal connected.
 * -------------------------------------------------------------------------- */

b_errtype BestSerCheckConnection(b_portnumtype OsHandle)
{
  int minfo;

  //printf("Serial: SerCheckConnection\n");

  if(OsHandle<0)
    return B_E_BAD_HANDLE;

  ioctl(OsHandle, TIOCMGET, &minfo);
 
  if ( minfo & TIOCM_DSR) 
    return B_E_OK;

  return B_E_NOT_CONNECTED;
}


/* --------------------------------------------------------------------------
 * Should only be called from BestReleaseConnection()
 * We clear our DTR (card's DSR) to signal disconnect.
 * -------------------------------------------------------------------------- */

void BestSerReleaseConnection(b_portnumtype OsHandle)
{
  int minfo;

  printf("Serial: ReleaseConnection\n");

  if(OsHandle < 0)
    return;

  ioctl(OsHandle,TIOCMGET,&minfo);
  minfo &= ~TIOCM_DTR;
  ioctl(OsHandle,TIOCMSET,&minfo);

  DBG_PRINT("\nSerial Port Disconnected");
}


/* --------------------------------------------------------------------------
 * Should only be called from BestOnReadError()
 * -------------------------------------------------------------------------- */

b_errtype BestSerOnReadOrWriteError(b_portnumtype OsHandle)
{
#ifdef BEST_DEBUG

#else
  printf("Serial: OnReadOrWriteError\n");

  if(OsHandle < 0)
    return B_E_BAD_HANDLE;

  return BestClearCommError(OsHandle);

#endif
}


/* --------------------------------------------------------------------------
 * Should only be called from BestPortTimeoutSet()
 * -------------------------------------------------------------------------- */
b_errtype BestSerPortTimeoutSet(b_portnumtype OsHandle, 
                                BESTTIMEOUTS * pCallersTimeouts)
{
  printf("Serial:PortTimeOutSet\n");
  return B_E_OK;
  return B_E_SETTING_TIMEOUTS;
  //  return SetCommTimeouts(OsHandle, (LPCOMMTIMEOUTS) pCallersTimeouts) ?
  //          B_E_OK : 
  //          B_E_SETTING_TIMEOUTS;
}


/******************************************************************************/

#define SER_IN_BUF_SIZE 4096
#define SER_OUT_BUF_SIZE 4096

b_portnumtype BestOpenCOMInitial(int num, b_int32 Baudrate)
{
  int OsHandle;
  int brate;
  struct termios termio;
  char DevName[20]; /* pointer to com port string    */

  printf("Serial: OpenCOMInitial\n");

  /* check for valid port number */
  if (num < 1 || num > 255)
  {
    return INVALID_OS_HANDLE;
  }

  sprintf (DevName, "/dev/ttyS%d", num-1); /* COM1 is /dev/ttys0 */
  OsHandle = open(DevName, O_RDWR|O_SYNC|O_NOCTTY);

  /* open the comport as a file */
  if( OsHandle < 0 )
  {
    //fprintf(stderr,"BestOpenCOMInitial failed: errno=%d\n",errno);
    return INVALID_OS_HANDLE;
  }

  /* check the baudrate */
  switch (Baudrate)
  {
  case 9600:
    brate = B9600;
    break;
  case 19200:
    brate = B19200;
    break;
  case 38400:
    brate = B38400;
    break;
  case 57600:
    brate = B57600;
    break;
  case 115200:
    brate = B115200;
    break;
  default:
    BestCloseSerial(OsHandle);
    return INVALID_OS_HANDLE;
  }

  /* No monitoring */
  //  (void) SetCommMask(OsHandle, 0);

  /* buffer sizes */
  //  (void) SetupComm(OsHandle, SER_IN_BUF_SIZE, SER_OUT_BUF_SIZE);

  tcgetattr(OsHandle,&termio);
  termio.c_iflag = IGNBRK | IGNPAR | IGNCR;
  termio.c_oflag = 0; // Raw output
  termio.c_cflag = CS8 | CREAD| CLOCAL | CRTSCTS | HUPCL;
  termio.c_lflag = 0;
  cfsetispeed(&termio,brate);
  cfsetospeed(&termio,brate);
  
  tcsetattr(OsHandle,TCSANOW,&termio);

  /* setup protocol and line control.
   * DTR (from PC) and DSR (from Card) are used to indicate "connected".
   * RTS/CTS used for hardware flow control.  
   * DSR sensitivity is used to help prevent junk in the input buffer.
   */
/*  (void) GetCommState(OsHandle, &dcb);

  dcb.DCBlength = sizeof(DCB);
  dcb.BaudRate = brate;
  dcb.fBinary = TRUE;
  dcb.fParity = FALSE;
  dcb.fOutxCtsFlow = TRUE;
  dcb.fOutxDsrFlow = TRUE; 
  dcb.fDtrControl = DTR_CONTROL_DISABLE;
  dcb.fDsrSensitivity = TRUE;
  dcb.fOutX = FALSE;
  dcb.fInX = FALSE;
  dcb.fRtsControl = RTS_CONTROL_HANDSHAKE;
  dcb.fAbortOnError = FALSE;
  dcb.ByteSize = 8;
  dcb.Parity = NOPARITY;
  dcb.StopBits = ONESTOPBIT;

  if (!SetCommState(OsHandle, &dcb))
  {
    BestCloseSerial(OsHandle);
    return INVALID_OS_HANDLE;
  }
*/

  BestSerReleaseConnection((b_portnumtype) OsHandle);
  return (b_portnumtype) OsHandle;
}


/******************************************************************************/
void BestCloseSerial(b_portnumtype OsHandle)
{
  printf("Serial: CloseSerial\n");
 
  BestSerReleaseConnection(OsHandle);
  close(OsHandle);
}
